// music box: predecessor program to a musical beeper
// to be used for model rocket recovery
// by Auren Amster, July 2024 <biz@aurenamster.com>

import processing.sound.*;
SinOsc sounder;
byte noteFile[];
String key = "";
int tempo;

int n = 0; // which character of file are we reading right now;
int lastInterval = 1; // the speed of the last note, useful for grouping


void setup() {
  size(640, 360);
  background(255);
    
  // Create the sine oscillator.
  sounder = new SinOsc(this);
  
  // load note data
  noteFile = loadBytes("notes.txt");
  
  // configure key signature, time signature, and tempo
  int i = 0;
  while(noteFile[i] != '\n') {
    key += char(noteFile[i]);
    i++;
  }
  i++; // next line
  String tempoStr = "";
  while(noteFile[i] != '\n') {
    tempoStr += char(noteFile[i]);
    i++;
  }
  tempo = int(tempoStr);
  println("key: " + key);
  println("tempo: " + tempo + " bpm");
  n = i + 1;
}

/*
  file format:
  c#               <-- key signature (relative major)
  60               <-- tempo
  c-7-4 d-7-4 ...  <-- begin notes
  
 - notes separated by one space or one \n

accidental (b/-/#)   interval (multiple digits supported, '*' for same as previous note)
                 \    /
    note format: c#6d4 
               /    \ \------------------\   
           note    octave [a4 = 440Hz]   dotted
 
 this is a c-sharp-6 quarter note.
 
 more examples:
 d-4-8 --> d4 eighth note
 bb2-2 --> b-flat-2 half note
 a-4-4 --> a4 quarter note
 e-7-16 --> e7 16th note
 f-5-* --> f5 16th note (same as above)
 g-3d2 --> g3 dotted half note

*/

void draw() {
  n = parseNote(noteFile, n);
}

char toChar(byte data) {
  int ascii = data & 0xff;
  return char(ascii);
}

// read given file to find the next note and a helper iterator
// return the updated value of i
int parseNote(byte file[], int i) {
  char letter = toChar(file[i]);
  char accidental = toChar(file[i + 1]);
  int octave = toChar(file[i + 2]) - 48;
  boolean dotted = toChar(file[i + 3]) == 'd' ? true : false;
  String intervalString = "";
  int x = i+4;
  while (file[x] != ' ' && file[x] != '\n') {
    intervalString += char(file[x]);
    x++;
  }
  int interval = 0;
  if (intervalString.equals("*")) {
    interval = lastInterval; // repeated note length
  } else {
    interval = int(intervalString);
    lastInterval = interval;
  }
  i += 4 + intervalString.length();
  
  play(letter, accidental, octave, dotted, interval);
  if (i + 1 >= file.length - 1) {
    exit(); // the end of the song has been reached
  } else {
    i++;
  }
  return i;
}

// supports notes a - g, and r for rest
void play(char letter, char accidental, int octave, boolean dotted, int interval) {
  println(letter + " " + accidental + " " + octave + " " + interval);
  //                     pure note tone             A4 reference pitch    octave offset
  if (letter != 'r') {
    float frequency = pow(1.059463, getAOffset(letter, accidental)) * 440 * pow(2, octave-4);
    sounder.freq(frequency);
    sounder.play();
  }
  int duration = round(60000 / tempo / interval * 4 * (dotted ? 1.5 : 1));
  delay(duration*1/2);
  sounder.stop();
  delay(duration*1/2);
}

int getAOffset(char letter, char accidental) {
  int noteOffset = 0;
  int accidentalOffset = 0;
  int signatureOffset = 0;
  
  switch (accidental) {
    case '#':
      accidentalOffset = 1;
      break;
    case 'b':
      accidentalOffset = -1;
      break;
  }
  
  switch (letter) {
    case 'c':
      noteOffset = -9;
      break;
    case 'd':
      noteOffset = -7;
      break;
    case 'e':
      noteOffset = -5;
      break;
    case 'f':
      noteOffset = -4;
      break;
    case 'g':
      noteOffset = -2;
      break;
    case 'a':
      noteOffset = 0;
      break;
    case 'b':
      noteOffset = 2;
      break;
  }
  
  switch (key) {
    case "c":
      break;
    case "db":
      switch (letter) {
        case 'd':
        case 'e':
        case 'g':
        case 'a':
        case 'b':
          signatureOffset = -1;
          break;
      }
      break;
    case "d":
      switch (letter) {
        case 'f':
        case 'c':
          signatureOffset = 1;
          break;
      }
      break;
    case "eb": {
      switch (letter) {
        case 'e':
        case 'a':
        case 'b':
          signatureOffset = -1;
          break;
      }
      break;
    }
  }
  
  return noteOffset + accidentalOffset + signatureOffset;
}
